const axios = require('axios');
const fs = require('fs-extra');
const FormData = require('form-data');
const crypto = require('crypto');
const path = require('path');
const config = require('../../config/config');
const AuthService = require('./AuthService');
const MapData = require('../models/MapData');

class UploadService {
  constructor() {
    this.authService = new AuthService();
    this.apiClient = axios.create({
      baseURL: `${config.api.baseUrl}/${config.api.version}`,
      timeout: config.api.timeout
    });
  }

  // Calculate MD5 hash of file
  async calculateMD5(filePath) {
    return new Promise((resolve, reject) => {
      const hash = crypto.createHash('md5');
      const stream = fs.createReadStream(filePath);
      
      stream.on('data', (data) => hash.update(data));
      stream.on('end', () => resolve(hash.digest('hex').toUpperCase()));
      stream.on('error', reject);
    });
  }

  // Get file info
  async getFileInfo(filePath) {
    const stats = await fs.stat(filePath);
    const md5Hash = await this.calculateMD5(filePath);
    const fileName = path.basename(filePath);
    const ext = path.extname(filePath).toLowerCase();
    
    // Determine data type based on file extension
    let type = 'unknown';
    if (['.fbl', '.fjv', '.fpa', '.fda'].includes(ext)) {
      type = 'map';
    } else if (ext === '.poi') {
      type = 'poi';
    } else if (['.3dc', '.3dl'].includes(ext)) {
      type = 'building';
    } else if (['.spc', '.spud'].includes(ext)) {
      type = 'speedcam';
    }

    return {
      fileName,
      filePath,
      fileSize: stats.size,
      md5Hash,
      type,
      extension: ext
    };
  }

  // Upload single file
  async uploadFile(filePath, options = {}) {
    try {
      // Ensure authentication
      await this.authService.ensureAuthenticated();

      // Get file info
      const fileInfo = await this.getFileInfo(filePath);
      
      // Create MapData object
      const mapData = new MapData({
        name: options.name || fileInfo.fileName,
        type: fileInfo.type,
        fileName: fileInfo.fileName,
        filePath: fileInfo.filePath,
        fileSize: fileInfo.fileSize,
        md5Hash: fileInfo.md5Hash,
        version: options.version || '1.0.0',
        region: options.region || 'Vietnam',
        metadata: options.metadata || {}
      });

      // Validate data
      const validation = mapData.validate();
      if (!validation.isValid) {
        throw new Error(`Validation failed: ${validation.errors.join(', ')}`);
      }

      // Check if file already exists on server
      const existingFile = await this.checkFileExists(mapData.md5Hash);
      if (existingFile) {
        console.log(`File ${fileInfo.fileName} already exists on server`);
        return { success: true, message: 'File already exists', data: existingFile };
      }

      // Upload file
      const formData = new FormData();
      formData.append('file', fs.createReadStream(filePath));
      formData.append('data', JSON.stringify(mapData.toApiFormat()));

      const response = await this.apiClient.post('/upload', formData, {
        headers: {
          ...formData.getHeaders(),
          ...this.authService.getAuthHeader()
        },
        onUploadProgress: (progressEvent) => {
          if (options.onProgress) {
            const percentCompleted = Math.round((progressEvent.loaded * 100) / progressEvent.total);
            options.onProgress(percentCompleted);
          }
        }
      });

      mapData.updateStatus('completed');
      return { success: true, data: response.data, mapData };

    } catch (error) {
      console.error(`Upload failed for ${filePath}:`, error.message);
      throw error;
    }
  }

  // Check if file exists on server
  async checkFileExists(md5Hash) {
    try {
      const response = await this.apiClient.get(`/files/check/${md5Hash}`, {
        headers: this.authService.getAuthHeader()
      });
      return response.data;
    } catch (error) {
      if (error.response && error.response.status === 404) {
        return null;
      }
      throw error;
    }
  }

  // Upload multiple files
  async uploadMultipleFiles(filePaths, options = {}) {
    const results = [];
    const errors = [];

    for (let i = 0; i < filePaths.length; i++) {
      try {
        console.log(`Uploading file ${i + 1}/${filePaths.length}: ${path.basename(filePaths[i])}`);
        const result = await this.uploadFile(filePaths[i], options);
        results.push(result);
      } catch (error) {
        console.error(`Failed to upload ${filePaths[i]}:`, error.message);
        errors.push({ file: filePaths[i], error: error.message });
      }
    }

    return { results, errors };
  }

  // Get upload status
  async getUploadStatus(uploadId) {
    try {
      const response = await this.apiClient.get(`/upload/status/${uploadId}`, {
        headers: this.authService.getAuthHeader()
      });
      return response.data;
    } catch (error) {
      throw new Error(`Failed to get upload status: ${error.message}`);
    }
  }

  // List uploaded files
  async listUploadedFiles(filters = {}) {
    try {
      const response = await this.apiClient.get('/files', {
        headers: this.authService.getAuthHeader(),
        params: filters
      });
      return response.data;
    } catch (error) {
      throw new Error(`Failed to list files: ${error.message}`);
    }
  }

  // Delete uploaded file
  async deleteFile(fileId) {
    try {
      const response = await this.apiClient.delete(`/files/${fileId}`, {
        headers: this.authService.getAuthHeader()
      });
      return response.data;
    } catch (error) {
      throw new Error(`Failed to delete file: ${error.message}`);
    }
  }
}

module.exports = UploadService;
