const fs = require('fs-extra');
const path = require('path');
const config = require('../../config/config');
const UploadService = require('./UploadService');

class DataSyncService {
  constructor() {
    this.uploadService = new UploadService();
    this.syncStatus = {
      lastSync: null,
      totalFiles: 0,
      uploadedFiles: 0,
      failedFiles: 0,
      skippedFiles: 0
    };
  }

  // Scan directory for map data files
  async scanDirectory(dirPath) {
    const files = [];
    
    try {
      const items = await fs.readdir(dirPath);
      
      for (const item of items) {
        const itemPath = path.join(dirPath, item);
        const stats = await fs.stat(itemPath);
        
        if (stats.isDirectory()) {
          // Recursively scan subdirectories
          const subFiles = await this.scanDirectory(itemPath);
          files.push(...subFiles);
        } else if (stats.isFile()) {
          const ext = path.extname(item).toLowerCase();
          if (config.upload.supportedFormats.includes(ext)) {
            files.push(itemPath);
          }
        }
      }
    } catch (error) {
      console.error(`Error scanning directory ${dirPath}:`, error.message);
    }
    
    return files;
  }

  // Get all map data files from project
  async getAllMapDataFiles() {
    const allFiles = [];
    const paths = config.paths;
    
    // Scan all data directories
    for (const [type, dirPath] of Object.entries(paths)) {
      if (type !== 'tempDir' && type !== 'backupDir') {
        try {
          const fullPath = path.resolve(dirPath);
          if (await fs.pathExists(fullPath)) {
            const files = await this.scanDirectory(fullPath);
            allFiles.push(...files.map(file => ({ file, type })));
          }
        } catch (error) {
          console.warn(`Could not scan ${type} directory:`, error.message);
        }
      }
    }
    
    return allFiles;
  }

  // Sync all data to server
  async syncAllData(options = {}) {
    console.log('Starting data synchronization...');
    this.syncStatus = {
      lastSync: new Date(),
      totalFiles: 0,
      uploadedFiles: 0,
      failedFiles: 0,
      skippedFiles: 0
    };

    try {
      // Get all files
      const allFiles = await this.getAllMapDataFiles();
      this.syncStatus.totalFiles = allFiles.length;
      
      console.log(`Found ${allFiles.length} files to sync`);

      // Group files by type
      const filesByType = {};
      allFiles.forEach(({ file, type }) => {
        if (!filesByType[type]) {
          filesByType[type] = [];
        }
        filesByType[type].push(file);
      });

      // Upload files by type
      for (const [type, files] of Object.entries(filesByType)) {
        console.log(`\nSyncing ${type} files (${files.length} files)...`);
        
        const uploadOptions = {
          type,
          region: 'Vietnam',
          version: '1.0.0',
          metadata: {
            syncDate: new Date().toISOString(),
            source: 'local-export'
          }
        };

        const result = await this.uploadService.uploadMultipleFiles(files, uploadOptions);
        
        this.syncStatus.uploadedFiles += result.results.length;
        this.syncStatus.failedFiles += result.errors.length;
        
        // Log results
        if (result.results.length > 0) {
          console.log(`✓ Successfully uploaded ${result.results.length} ${type} files`);
        }
        
        if (result.errors.length > 0) {
          console.log(`✗ Failed to upload ${result.errors.length} ${type} files`);
          result.errors.forEach(error => {
            console.log(`  - ${path.basename(error.file)}: ${error.error}`);
          });
        }
      }

      // Save sync status
      await this.saveSyncStatus();
      
      console.log('\n=== Sync Summary ===');
      console.log(`Total files: ${this.syncStatus.totalFiles}`);
      console.log(`Uploaded: ${this.syncStatus.uploadedFiles}`);
      console.log(`Failed: ${this.syncStatus.failedFiles}`);
      console.log(`Skipped: ${this.syncStatus.skippedFiles}`);
      console.log(`Last sync: ${this.syncStatus.lastSync}`);

      return this.syncStatus;

    } catch (error) {
      console.error('Sync failed:', error.message);
      throw error;
    }
  }

  // Save sync status to file
  async saveSyncStatus() {
    const statusFile = path.join(config.paths.tempDir, 'sync-status.json');
    await fs.ensureDir(path.dirname(statusFile));
    await fs.writeJson(statusFile, this.syncStatus, { spaces: 2 });
  }

  // Load sync status from file
  async loadSyncStatus() {
    const statusFile = path.join(config.paths.tempDir, 'sync-status.json');
    
    if (await fs.pathExists(statusFile)) {
      this.syncStatus = await fs.readJson(statusFile);
    }
    
    return this.syncStatus;
  }

  // Check for new or modified files
  async checkForChanges() {
    const currentFiles = await this.getAllMapDataFiles();
    const lastSync = this.syncStatus.lastSync;
    
    if (!lastSync) {
      return { hasChanges: true, newFiles: currentFiles };
    }

    const newFiles = [];
    const modifiedFiles = [];

    for (const { file } of currentFiles) {
      const stats = await fs.stat(file);
      if (stats.mtime > new Date(lastSync)) {
        newFiles.push(file);
      }
    }

    return {
      hasChanges: newFiles.length > 0,
      newFiles,
      modifiedFiles
    };
  }

  // Incremental sync (only new/modified files)
  async incrementalSync() {
    console.log('Checking for changes...');
    
    const changes = await this.checkForChanges();
    
    if (!changes.hasChanges) {
      console.log('No changes detected. Data is up to date.');
      return { success: true, message: 'No changes to sync' };
    }

    console.log(`Found ${changes.newFiles.length} new/modified files`);
    return await this.syncAllData();
  }
}

module.exports = DataSyncService;
