const { Command } = require('commander');
const chalk = require('chalk');
const inquirer = require('inquirer');
const DataSyncService = require('./services/DataSyncService');
const UploadService = require('./services/UploadService');

const program = new Command();

program
  .name('zzmaps-api')
  .description('ZZMaps API Client for map data management')
  .version('1.0.0');

// Main menu command
program
  .command('menu')
  .description('Show interactive menu')
  .action(async () => {
    await showMainMenu();
  });

// Quick upload command
program
  .command('upload')
  .description('Quick upload all data')
  .option('-i, --incremental', 'Incremental sync only')
  .action(async (options) => {
    try {
      const dataSyncService = new DataSyncService();
      
      console.log(chalk.blue('Starting data upload...'));
      
      if (options.incremental) {
        await dataSyncService.incrementalSync();
      } else {
        await dataSyncService.syncAllData();
      }
      
      console.log(chalk.green('Upload completed!'));
      
    } catch (error) {
      console.error(chalk.red('Upload failed:'), error.message);
      process.exit(1);
    }
  });

// Interactive main menu
async function showMainMenu() {
  console.log(chalk.blue.bold('\n=== ZZMaps API Client ===\n'));
  
  while (true) {
    const { action } = await inquirer.prompt([{
      type: 'list',
      name: 'action',
      message: 'What would you like to do?',
      choices: [
        { name: '📤 Upload all data', value: 'upload-all' },
        { name: '🔄 Incremental sync', value: 'incremental-sync' },
        { name: '📋 List uploaded files', value: 'list-files' },
        { name: '📊 Show sync status', value: 'show-status' },
        { name: '⚙️  Configure settings', value: 'configure' },
        { name: '❌ Exit', value: 'exit' }
      ]
    }]);
    
    try {
      switch (action) {
        case 'upload-all':
          await handleUploadAll();
          break;
        case 'incremental-sync':
          await handleIncrementalSync();
          break;
        case 'list-files':
          await handleListFiles();
          break;
        case 'show-status':
          await handleShowStatus();
          break;
        case 'configure':
          await handleConfigure();
          break;
        case 'exit':
          console.log(chalk.yellow('Goodbye!'));
          process.exit(0);
      }
    } catch (error) {
      console.error(chalk.red('Error:'), error.message);
    }
    
    console.log(''); // Add spacing
  }
}

// Handle upload all data
async function handleUploadAll() {
  const { confirm } = await inquirer.prompt([{
    type: 'confirm',
    name: 'confirm',
    message: 'This will upload all map data files. Continue?',
    default: true
  }]);
  
  if (!confirm) return;
  
  const dataSyncService = new DataSyncService();
  const result = await dataSyncService.syncAllData();
  
  console.log(chalk.green('\nUpload completed!'));
  console.log(`Uploaded: ${result.uploadedFiles} files`);
  console.log(`Failed: ${result.failedFiles} files`);
}

// Handle incremental sync
async function handleIncrementalSync() {
  const dataSyncService = new DataSyncService();
  const result = await dataSyncService.incrementalSync();
  
  if (result.success && result.message === 'No changes to sync') {
    console.log(chalk.green('No changes detected. Data is up to date.'));
  } else {
    console.log(chalk.green('Incremental sync completed!'));
    console.log(`Uploaded: ${result.uploadedFiles} files`);
    console.log(`Failed: ${result.failedFiles} files`);
  }
}

// Handle list files
async function handleListFiles() {
  const uploadService = new UploadService();
  const files = await uploadService.listUploadedFiles();
  
  if (files.length === 0) {
    console.log(chalk.yellow('No files found on server'));
    return;
  }
  
  console.log(chalk.blue('\n=== Uploaded Files ==='));
  files.forEach((file, index) => {
    console.log(`${index + 1}. ${file.fileName}`);
    console.log(`   Type: ${file.type} | Size: ${formatBytes(file.fileSize)}`);
    console.log(`   Uploaded: ${new Date(file.uploadedAt).toLocaleString()}`);
    console.log('');
  });
}

// Handle show status
async function handleShowStatus() {
  const dataSyncService = new DataSyncService();
  const status = await dataSyncService.loadSyncStatus();
  
  console.log(chalk.blue('\n=== Sync Status ==='));
  console.log(`Last sync: ${status.lastSync ? new Date(status.lastSync).toLocaleString() : 'Never'}`);
  console.log(`Total files: ${status.totalFiles}`);
  console.log(`Uploaded: ${status.uploadedFiles}`);
  console.log(`Failed: ${status.failedFiles}`);
  console.log(`Skipped: ${status.skippedFiles}`);
}

// Handle configure
async function handleConfigure() {
  console.log(chalk.yellow('\nConfiguration options:'));
  console.log('1. Set API credentials');
  console.log('2. Configure upload settings');
  console.log('3. View current configuration');
  
  const { option } = await inquirer.prompt([{
    type: 'list',
    name: 'option',
    message: 'Select configuration option:',
    choices: [
      { name: 'Set API credentials', value: 'credentials' },
      { name: 'Configure upload settings', value: 'upload' },
      { name: 'View current configuration', value: 'view' },
      { name: 'Back to main menu', value: 'back' }
    ]
  }]);
  
  switch (option) {
    case 'credentials':
      await handleConfigureCredentials();
      break;
    case 'upload':
      await handleConfigureUpload();
      break;
    case 'view':
      await handleViewConfiguration();
      break;
    case 'back':
      return;
  }
}

// Handle configure credentials
async function handleConfigureCredentials() {
  console.log(chalk.yellow('\nTo configure API credentials, set the following environment variables:'));
  console.log('export ZZMAPS_USERNAME="your_username"');
  console.log('export ZZMAPS_PASSWORD="your_password"');
  console.log('export ZZMAPS_API_KEY="your_api_key"');
  console.log('\nOr edit the config file: config/config.js');
}

// Handle configure upload
async function handleConfigureUpload() {
  console.log(chalk.yellow('\nUpload configuration is in: config/config.js'));
  console.log('You can modify:');
  console.log('- Chunk size for uploads');
  console.log('- Maximum concurrent uploads');
  console.log('- Supported file formats');
  console.log('- Maximum file size');
}

// Handle view configuration
async function handleViewConfiguration() {
  const config = require('../config/config');
  
  console.log(chalk.blue('\n=== Current Configuration ==='));
  console.log(`API Base URL: ${config.api.baseUrl}`);
  console.log(`API Version: ${config.api.version}`);
  console.log(`Username: ${config.auth.username || 'Not set'}`);
  console.log(`API Key: ${config.auth.apiKey ? 'Set' : 'Not set'}`);
  console.log(`Chunk Size: ${formatBytes(config.upload.chunkSize)}`);
  console.log(`Max Concurrent Uploads: ${config.upload.maxConcurrentUploads}`);
  console.log(`Max File Size: ${formatBytes(config.upload.maxFileSize)}`);
  console.log(`Supported Formats: ${config.upload.supportedFormats.join(', ')}`);
}

// Helper function to format bytes
function formatBytes(bytes, decimals = 2) {
  if (bytes === 0) return '0 Bytes';
  
  const k = 1024;
  const dm = decimals < 0 ? 0 : decimals;
  const sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB'];
  
  const i = Math.floor(Math.log(bytes) / Math.log(k));
  
  return parseFloat((bytes / Math.pow(k, i)).toFixed(dm)) + ' ' + sizes[i];
}

// Error handling
process.on('unhandledRejection', (reason, promise) => {
  console.error(chalk.red('Unhandled Rejection at:'), promise, chalk.red('reason:'), reason);
  process.exit(1);
});

// Parse command line arguments
program.parse();
