#!/bin/bash

echo "========================================"
echo "   ZZMaps API Client - Shared Hosting"
echo "========================================"
echo

# Check if running on shared hosting
echo "Detecting hosting environment..."

# Check for common shared hosting indicators
if [ -d "/home" ] && [ -d "/var/www" ]; then
    echo "✓ Shared hosting environment detected"
    HOSTING_TYPE="shared"
elif [ -d "/home" ] && [ ! -d "/var/www" ]; then
    echo "✓ VPS/Dedicated server detected"
    HOSTING_TYPE="vps"
else
    echo "✓ Local development environment detected"
    HOSTING_TYPE="local"
fi

echo "Hosting type: $HOSTING_TYPE"
echo

# Check if Node.js is installed
if ! command -v node &> /dev/null; then
    echo "Node.js not found. Installing..."
    
    if [ "$HOSTING_TYPE" = "shared" ]; then
        echo "Shared hosting detected. Installing Node.js via NodeSource..."
        
        # Install Node.js via NodeSource (works on shared hosting)
        curl -fsSL https://deb.nodesource.com/setup_18.x | sudo -E bash -
        sudo apt-get install -y nodejs
        
    else
        # Regular installation for VPS/local
        sudo apt update
        sudo apt install -y nodejs npm
    fi
    
    # Check installation
    if ! command -v node &> /dev/null; then
        echo "ERROR: Failed to install Node.js!"
        echo "Please contact your hosting provider for Node.js support"
        exit 1
    fi
fi

echo "Node.js is installed:"
node --version
echo

# Check if npm is installed
if ! command -v npm &> /dev/null; then
    echo "ERROR: npm is not installed!"
    echo "Please install npm: sudo apt install npm"
    exit 1
fi

echo "npm is installed:"
npm --version
echo

# Install dependencies
echo "Installing dependencies..."
npm install
if [ $? -ne 0 ]; then
    echo "ERROR: Failed to install dependencies!"
    echo "This might be due to shared hosting restrictions."
    echo "Please contact your hosting provider for Node.js support"
    exit 1
fi

echo
echo "Dependencies installed successfully!"
echo

# Create .env file if it doesn't exist
if [ ! -f .env ]; then
    echo "Creating .env file..."
    cp env.example .env
    echo
    echo "IMPORTANT: Please edit .env file with your API credentials!"
    echo "You can edit it with: nano .env"
    echo
fi

# Run setup
echo "Running project setup..."
npm run setup
if [ $? -ne 0 ]; then
    echo "ERROR: Setup failed!"
    exit 1
fi

echo
echo "========================================"
echo "   Installation completed successfully!"
echo "========================================"
echo
echo "Hosting environment: $HOSTING_TYPE"
echo "Node.js version: $(node --version)"
echo "npm version: $(npm --version)"
echo
echo "Available commands:"
echo "  npm start                    - Interactive menu"
echo "  npm run upload-current      - Upload current data"
echo "  npm run setup               - Setup project"
echo "  node scripts/demo.js        - Demo project"
echo
echo "To upload data:"
echo "  ./upload.sh"
echo
echo "To run interactive menu:"
echo "  ./run.sh"
echo
echo "Press any key to continue..."
read -n 1
