#!/usr/bin/env node

const { Command } = require('commander');
const chalk = require('chalk');
const ora = require('ora');
const inquirer = require('inquirer');
const DataSyncService = require('../src/services/DataSyncService');
const config = require('../config/config');

const program = new Command();

program
  .name('sync-data')
  .description('Synchronize map data with zzmaps.com API')
  .version('1.0.0');

// Full sync command
program
  .command('full')
  .description('Perform full synchronization of all data')
  .option('-f, --force', 'Force sync even if no changes detected')
  .action(async (options) => {
    const spinner = ora('Starting full sync...').start();
    
    try {
      const dataSyncService = new DataSyncService();
      
      // Check authentication
      spinner.text = 'Authenticating...';
      await dataSyncService.uploadService.authService.ensureAuthenticated();
      
      // Load previous sync status
      spinner.text = 'Loading sync status...';
      await dataSyncService.loadSyncStatus();
      
      // Check for changes
      if (!options.force) {
        spinner.text = 'Checking for changes...';
        const changes = await dataSyncService.checkForChanges();
        
        if (!changes.hasChanges) {
          spinner.succeed(chalk.green('No changes detected. Data is up to date.'));
          return;
        }
        
        console.log(chalk.yellow(`Found ${changes.newFiles.length} new/modified files`));
      }
      
      // Confirm sync
      spinner.stop();
      const { confirm } = await inquirer.prompt([{
        type: 'confirm',
        name: 'confirm',
        message: 'Do you want to proceed with full synchronization?',
        default: true
      }]);
      
      if (!confirm) {
        console.log(chalk.yellow('Sync cancelled'));
        return;
      }
      
      // Perform sync
      spinner.start('Synchronizing data...');
      const result = await dataSyncService.syncAllData();
      
      spinner.succeed(chalk.green('Full sync completed!'));
      
      // Display summary
      displaySyncSummary(result);
      
    } catch (error) {
      spinner.fail(chalk.red('Sync failed!'));
      console.error(chalk.red('Error:'), error.message);
      process.exit(1);
    }
  });

// Incremental sync command
program
  .command('incremental')
  .description('Perform incremental sync (only new/modified files)')
  .action(async () => {
    const spinner = ora('Starting incremental sync...').start();
    
    try {
      const dataSyncService = new DataSyncService();
      
      // Check authentication
      spinner.text = 'Authenticating...';
      await dataSyncService.uploadService.authService.ensureAuthenticated();
      
      // Load previous sync status
      spinner.text = 'Loading sync status...';
      await dataSyncService.loadSyncStatus();
      
      // Check for changes
      spinner.text = 'Checking for changes...';
      const changes = await dataSyncService.checkForChanges();
      
      if (!changes.hasChanges) {
        spinner.succeed(chalk.green('No changes detected. Data is up to date.'));
        return;
      }
      
      console.log(chalk.yellow(`Found ${changes.newFiles.length} new/modified files`));
      
      // Confirm sync
      spinner.stop();
      const { confirm } = await inquirer.prompt([{
        type: 'confirm',
        name: 'confirm',
        message: `Do you want to sync ${changes.newFiles.length} new/modified files?`,
        default: true
      }]);
      
      if (!confirm) {
        console.log(chalk.yellow('Sync cancelled'));
        return;
      }
      
      // Perform incremental sync
      spinner.start('Synchronizing changes...');
      const result = await dataSyncService.incrementalSync();
      
      if (result.success && result.message === 'No changes to sync') {
        spinner.succeed(chalk.green('No changes to sync'));
        return;
      }
      
      spinner.succeed(chalk.green('Incremental sync completed!'));
      
      // Display summary
      displaySyncSummary(result);
      
    } catch (error) {
      spinner.fail(chalk.red('Sync failed!'));
      console.error(chalk.red('Error:'), error.message);
      process.exit(1);
    }
  });

// Status command
program
  .command('status')
  .description('Show sync status and statistics')
  .action(async () => {
    const spinner = ora('Loading sync status...').start();
    
    try {
      const dataSyncService = new DataSyncService();
      const status = await dataSyncService.loadSyncStatus();
      
      spinner.succeed(chalk.green('Sync status loaded!'));
      
      console.log('\n' + chalk.blue('=== Sync Status ==='));
      console.log(`Last sync: ${status.lastSync ? new Date(status.lastSync).toLocaleString() : 'Never'}`);
      console.log(`Total files: ${status.totalFiles}`);
      console.log(`Uploaded: ${chalk.green(status.uploadedFiles)}`);
      console.log(`Failed: ${chalk.red(status.failedFiles)}`);
      console.log(`Skipped: ${chalk.yellow(status.skippedFiles)}`);
      
      // Check for changes
      spinner.start('Checking for changes...');
      const changes = await dataSyncService.checkForChanges();
      spinner.stop();
      
      if (changes.hasChanges) {
        console.log(`\n${chalk.yellow('Changes detected:')} ${changes.newFiles.length} new/modified files`);
        console.log('Run "sync-data incremental" to sync changes');
      } else {
        console.log(`\n${chalk.green('No changes detected. Data is up to date.')}`);
      }
      
    } catch (error) {
      spinner.fail(chalk.red('Failed to load status!'));
      console.error(chalk.red('Error:'), error.message);
      process.exit(1);
    }
  });

// Reset command
program
  .command('reset')
  .description('Reset sync status (will force full sync on next run)')
  .option('-f, --force', 'Skip confirmation')
  .action(async (options) => {
    try {
      if (!options.force) {
        const { confirm } = await inquirer.prompt([{
          type: 'confirm',
          name: 'confirm',
          message: 'Are you sure you want to reset sync status? This will force a full sync on next run.',
          default: false
        }]);
        
        if (!confirm) {
          console.log(chalk.yellow('Reset cancelled'));
          return;
        }
      }
      
      const spinner = ora('Resetting sync status...').start();
      const dataSyncService = new DataSyncService();
      
      // Reset status
      dataSyncService.syncStatus = {
        lastSync: null,
        totalFiles: 0,
        uploadedFiles: 0,
        failedFiles: 0,
        skippedFiles: 0
      };
      
      await dataSyncService.saveSyncStatus();
      
      spinner.succeed(chalk.green('Sync status reset!'));
      console.log(chalk.yellow('Next sync will be a full synchronization'));
      
    } catch (error) {
      spinner.fail(chalk.red('Reset failed!'));
      console.error(chalk.red('Error:'), error.message);
      process.exit(1);
    }
  });

// Helper function to display sync summary
function displaySyncSummary(result) {
  console.log('\n' + chalk.blue('=== Sync Summary ==='));
  console.log(`Total files: ${result.totalFiles}`);
  console.log(`Uploaded: ${chalk.green(result.uploadedFiles)}`);
  console.log(`Failed: ${chalk.red(result.failedFiles)}`);
  console.log(`Skipped: ${chalk.yellow(result.skippedFiles)}`);
  console.log(`Last sync: ${result.lastSync ? new Date(result.lastSync).toLocaleString() : 'Never'}`);
  
  if (result.failedFiles > 0) {
    console.log(chalk.yellow('\nSome files failed to upload. Check the logs for details.'));
  }
}

// Error handling
process.on('unhandledRejection', (reason, promise) => {
  console.error(chalk.red('Unhandled Rejection at:'), promise, chalk.red('reason:'), reason);
  process.exit(1);
});

// Parse command line arguments
program.parse();
