#!/usr/bin/env node

const chalk = require('chalk');
const ora = require('ora');
const fs = require('fs-extra');
const path = require('path');
const inquirer = require('inquirer');

async function setupProject() {
  console.log(chalk.blue.bold('\n=== ZZMaps API Project Setup ===\n'));
  
  try {
    // Check if we're in the right directory
    const currentDir = process.cwd();
    const parentDir = path.dirname(currentDir);
    
    console.log(chalk.yellow(`Current directory: ${currentDir}`));
    console.log(chalk.yellow(`Parent directory: ${parentDir}`));
    
    // Check for map data in parent directory
    const mapDir = path.join(parentDir, 'map');
    const poiDir = path.join(parentDir, 'poi');
    const buildingDir = path.join(parentDir, 'building');
    const speedcamDir = path.join(parentDir, 'speedcam');
    
    const dirsExist = {
      map: await fs.pathExists(mapDir),
      poi: await fs.pathExists(poiDir),
      building: await fs.pathExists(buildingDir),
      speedcam: await fs.pathExists(speedcamDir)
    };
    
    console.log(chalk.blue('\nChecking for map data directories:'));
    Object.entries(dirsExist).forEach(([dir, exists]) => {
      console.log(`  ${dir}: ${exists ? chalk.green('✓ Found') : chalk.red('✗ Not found')}`);
    });
    
    const hasMapData = Object.values(dirsExist).some(exists => exists);
    
    if (!hasMapData) {
      console.log(chalk.red('\nNo map data directories found in parent directory!'));
      console.log(chalk.yellow('Please ensure the following directories exist:'));
      console.log('  - map/');
      console.log('  - poi/');
      console.log('  - building/');
      console.log('  - speedcam/');
      process.exit(1);
    }
    
    // Create necessary directories
    const spinner = ora('Creating project directories...').start();
    
    const dirsToCreate = [
      'temp',
      'backup',
      'logs'
    ];
    
    for (const dir of dirsToCreate) {
      await fs.ensureDir(path.join(currentDir, dir));
    }
    
    spinner.succeed(chalk.green('Project directories created'));
    
    // Create .env file if it doesn't exist
    const envFile = path.join(currentDir, '.env');
    const envExampleFile = path.join(currentDir, 'env.example');
    
    if (!await fs.pathExists(envFile)) {
      if (await fs.pathExists(envExampleFile)) {
        await fs.copy(envExampleFile, envFile);
        console.log(chalk.green('Created .env file from template'));
      }
    }
    
    // Ask for API credentials
    console.log(chalk.blue('\n=== API Configuration ==='));
    console.log(chalk.yellow('Please provide your ZZMaps API credentials:'));
    
    const credentials = await inquirer.prompt([
      {
        type: 'input',
        name: 'username',
        message: 'Username:',
        validate: input => input.length > 0 || 'Username is required'
      },
      {
        type: 'password',
        name: 'password',
        message: 'Password:',
        validate: input => input.length > 0 || 'Password is required'
      },
      {
        type: 'input',
        name: 'apiKey',
        message: 'API Key (optional):',
        default: ''
      }
    ]);
    
    // Update .env file
    const envContent = `# ZZMaps API Configuration
ZZMAPS_USERNAME=${credentials.username}
ZZMAPS_PASSWORD=${credentials.password}
ZZMAPS_API_KEY=${credentials.apiKey}

# Optional: Override default settings
# API_BASE_URL=https://api.zzmaps.com
# API_VERSION=v1
# UPLOAD_CHUNK_SIZE=1048576
# MAX_CONCURRENT_UPLOADS=3
`;
    
    await fs.writeFile(envFile, envContent);
    console.log(chalk.green('Updated .env file with credentials'));
    
    // Count files
    spinner.start('Counting map data files...');
    
    const fileCounts = {};
    for (const [dir, exists] of Object.entries(dirsExist)) {
      if (exists) {
        const dirPath = path.join(parentDir, dir);
        const files = await fs.readdir(dirPath);
        const dataFiles = files.filter(file => 
          ['.fbl', '.fjv', '.fpa', '.fda', '.poi', '.3dc', '.3dl', '.spc', '.spud'].includes(path.extname(file).toLowerCase())
        );
        fileCounts[dir] = dataFiles.length;
      } else {
        fileCounts[dir] = 0;
      }
    }
    
    spinner.succeed(chalk.green('File count completed'));
    
    // Display summary
    console.log(chalk.blue('\n=== Setup Summary ==='));
    console.log(`Project directory: ${currentDir}`);
    console.log(`Map data directory: ${parentDir}`);
    console.log('\nFile counts:');
    Object.entries(fileCounts).forEach(([dir, count]) => {
      console.log(`  ${dir}: ${count} files`);
    });
    
    const totalFiles = Object.values(fileCounts).reduce((sum, count) => sum + count, 0);
    console.log(`\nTotal files: ${chalk.green(totalFiles)}`);
    
    // Ask if user wants to upload now
    const { uploadNow } = await inquirer.prompt([{
      type: 'confirm',
      name: 'uploadNow',
      message: 'Do you want to upload the data now?',
      default: true
    }]);
    
    if (uploadNow) {
      console.log(chalk.blue('\nStarting upload process...'));
      
      // Import and run upload
      const { execSync } = require('child_process');
      try {
        execSync('node scripts/upload-current-data.js', { stdio: 'inherit' });
      } catch (error) {
        console.log(chalk.yellow('\nUpload failed. You can try again later with:'));
        console.log(chalk.cyan('node scripts/upload-current-data.js'));
      }
    } else {
      console.log(chalk.yellow('\nSetup completed! You can upload data later with:'));
      console.log(chalk.cyan('node scripts/upload-current-data.js'));
    }
    
    console.log(chalk.green('\n=== Setup Completed Successfully! ==='));
    console.log(chalk.blue('\nAvailable commands:'));
    console.log('  npm start                    - Interactive menu');
    console.log('  npm run upload              - Quick upload all data');
    console.log('  node scripts/upload-current-data.js - Upload current data');
    console.log('  node scripts/sync-data.js   - Sync management');
    console.log('  node scripts/upload-data.js - File management');
    
  } catch (error) {
    console.error(chalk.red('\nSetup failed:'), error.message);
    process.exit(1);
  }
}

// Error handling
process.on('unhandledRejection', (reason, promise) => {
  console.error(chalk.red('Unhandled Rejection at:'), promise, chalk.red('reason:'), reason);
  process.exit(1);
});

// Run setup
setupProject();
