#!/usr/bin/env node

const chalk = require('chalk');
const ora = require('ora');
const path = require('path');
const fs = require('fs-extra');
const DataSyncService = require('../src/services/DataSyncService');

async function runDemo() {
  console.log(chalk.blue.bold('\n=== ZZMaps API Demo ===\n'));
  
  try {
    // Check if we're in the right directory
    const currentDir = process.cwd();
    const parentDir = path.dirname(currentDir);
    
    console.log(chalk.yellow(`Current directory: ${currentDir}`));
    console.log(chalk.yellow(`Parent directory: ${parentDir}`));
    
    // Check for map data
    const mapDir = path.join(parentDir, 'map');
    const poiDir = path.join(parentDir, 'poi');
    const buildingDir = path.join(parentDir, 'building');
    const speedcamDir = path.join(parentDir, 'speedcam');
    
    const dirsExist = {
      map: await fs.pathExists(mapDir),
      poi: await fs.pathExists(poiDir),
      building: await fs.pathExists(buildingDir),
      speedcam: await fs.pathExists(speedcamDir)
    };
    
    console.log(chalk.blue('\nChecking data directories:'));
    Object.entries(dirsExist).forEach(([dir, exists]) => {
      console.log(`  ${dir}: ${exists ? chalk.green('✓') : chalk.red('✗')}`);
    });
    
    // Count files
    const fileCounts = {};
    for (const [dir, exists] of Object.entries(dirsExist)) {
      if (exists) {
        const dirPath = path.join(parentDir, dir);
        const files = await fs.readdir(dirPath);
        const dataFiles = files.filter(file => 
          ['.fbl', '.fjv', '.fpa', '.fda', '.poi', '.3dc', '.3dl', '.spc', '.spud'].includes(path.extname(file).toLowerCase())
        );
        fileCounts[dir] = dataFiles.length;
      } else {
        fileCounts[dir] = 0;
      }
    }
    
    console.log(chalk.blue('\nFile counts:'));
    Object.entries(fileCounts).forEach(([dir, count]) => {
      console.log(`  ${dir}: ${count} files`);
    });
    
    const totalFiles = Object.values(fileCounts).reduce((sum, count) => sum + count, 0);
    console.log(chalk.green(`\nTotal files: ${totalFiles}`));
    
    // Show file details
    console.log(chalk.blue('\nFile details:'));
    for (const [dir, exists] of Object.entries(dirsExist)) {
      if (exists) {
        const dirPath = path.join(parentDir, dir);
        const files = await fs.readdir(dirPath);
        const dataFiles = files.filter(file => 
          ['.fbl', '.fjv', '.fpa', '.fda', '.poi', '.3dc', '.3dl', '.spc', '.spud'].includes(path.extname(file).toLowerCase())
        );
        
        if (dataFiles.length > 0) {
          console.log(`\n${chalk.cyan(dir.toUpperCase())}:`);
          for (const file of dataFiles) {
            const filePath = path.join(dirPath, file);
            const stats = await fs.stat(filePath);
            const size = formatBytes(stats.size);
            console.log(`  ${file} (${size})`);
          }
        }
      }
    }
    
    // Show configuration
    console.log(chalk.blue('\nConfiguration:'));
    const config = require('../config/config');
    console.log(`  API Base URL: ${config.api.baseUrl}`);
    console.log(`  API Version: ${config.api.version}`);
    console.log(`  Username: ${config.auth.username || 'Not set'}`);
    console.log(`  API Key: ${config.auth.apiKey ? 'Set' : 'Not set'}`);
    console.log(`  Chunk Size: ${formatBytes(config.upload.chunkSize)}`);
    console.log(`  Max Concurrent Uploads: ${config.upload.maxConcurrentUploads}`);
    console.log(`  Max File Size: ${formatBytes(config.upload.maxFileSize)}`);
    
    // Show available commands
    console.log(chalk.blue('\nAvailable commands:'));
    console.log('  npm start                    - Interactive menu');
    console.log('  npm run upload-current      - Upload current data');
    console.log('  npm run setup               - Setup project');
    console.log('  node scripts/sync-data.js   - Sync management');
    console.log('  node scripts/upload-data.js - File management');
    
    console.log(chalk.green('\nDemo completed successfully!'));
    
  } catch (error) {
    console.error(chalk.red('\nDemo failed:'), error.message);
    process.exit(1);
  }
}

// Helper function to format bytes
function formatBytes(bytes, decimals = 2) {
  if (bytes === 0) return '0 Bytes';
  
  const k = 1024;
  const dm = decimals < 0 ? 0 : decimals;
  const sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB'];
  
  const i = Math.floor(Math.log(bytes) / Math.log(k));
  
  return parseFloat((bytes / Math.pow(k, i)).toFixed(dm)) + ' ' + sizes[i];
}

// Error handling
process.on('unhandledRejection', (reason, promise) => {
  console.error(chalk.red('Unhandled Rejection at:'), promise, chalk.red('reason:'), reason);
  process.exit(1);
});

// Run demo
runDemo();

