const axios = require('axios');
const config = require('../../config/config');

class AuthService {
  constructor() {
    this.token = null;
    this.tokenExpiry = null;
    this.apiClient = axios.create({
      baseURL: `${config.api.baseUrl}/${config.api.version}`,
      timeout: config.api.timeout
    });
  }

  // Authenticate with API
  async authenticate() {
    try {
      const response = await this.apiClient.post('/auth/login', {
        username: config.auth.username,
        password: config.auth.password,
        apiKey: config.auth.apiKey
      });

      if (response.data && response.data.token) {
        this.token = response.data.token;
        this.tokenExpiry = new Date(response.data.expiresAt || Date.now() + 3600000); // 1 hour default
        return true;
      }
      
      throw new Error('Invalid authentication response');
    } catch (error) {
      console.error('Authentication failed:', error.message);
      throw new Error(`Authentication failed: ${error.message}`);
    }
  }

  // Check if token is valid
  isTokenValid() {
    if (!this.token || !this.tokenExpiry) {
      return false;
    }
    return new Date() < this.tokenExpiry;
  }

  // Get authorization header
  getAuthHeader() {
    if (!this.isTokenValid()) {
      throw new Error('No valid authentication token');
    }
    return {
      'Authorization': `Bearer ${this.token}`,
      'Content-Type': 'application/json'
    };
  }

  // Refresh token if needed
  async ensureAuthenticated() {
    if (!this.isTokenValid()) {
      await this.authenticate();
    }
    return this.token;
  }

  // Logout
  async logout() {
    try {
      if (this.token) {
        await this.apiClient.post('/auth/logout', {}, {
          headers: this.getAuthHeader()
        });
      }
    } catch (error) {
      console.warn('Logout failed:', error.message);
    } finally {
      this.token = null;
      this.tokenExpiry = null;
    }
  }
}

module.exports = AuthService;
