#!/usr/bin/env node

const { Command } = require('commander');
const chalk = require('chalk');
const ora = require('ora');
const inquirer = require('inquirer');
const path = require('path');
const fs = require('fs-extra');
const config = require('../config/config');
const UploadService = require('../src/services/UploadService');
const DataSyncService = require('../src/services/DataSyncService');

const program = new Command();

program
  .name('upload-data')
  .description('Upload map data to zzmaps.com API')
  .version('1.0.0');

// Upload single file command
program
  .command('file <filePath>')
  .description('Upload a single file')
  .option('-n, --name <name>', 'Custom name for the file')
  .option('-t, --type <type>', 'Data type (map, poi, building, speedcam)')
  .option('-v, --version <version>', 'Version number', '1.0.0')
  .option('-r, --region <region>', 'Region', 'Vietnam')
  .action(async (filePath, options) => {
    const spinner = ora('Uploading file...').start();
    
    try {
      const uploadService = new UploadService();
      
      const uploadOptions = {
        name: options.name,
        type: options.type,
        version: options.version,
        region: options.region,
        onProgress: (percent) => {
          spinner.text = `Uploading... ${percent}%`;
        }
      };
      
      const result = await uploadService.uploadFile(filePath, uploadOptions);
      
      spinner.succeed(chalk.green('File uploaded successfully!'));
      console.log(chalk.blue('Result:'), result);
      
    } catch (error) {
      spinner.fail(chalk.red('Upload failed!'));
      console.error(chalk.red('Error:'), error.message);
      process.exit(1);
    }
  });

// Upload all data command
program
  .command('all')
  .description('Upload all map data files')
  .option('-f, --force', 'Force upload even if files exist')
  .option('-i, --incremental', 'Only upload new/modified files')
  .action(async (options) => {
    const spinner = ora('Preparing upload...').start();
    
    try {
      const dataSyncService = new DataSyncService();
      
      // Check authentication
      spinner.text = 'Authenticating...';
      await dataSyncService.uploadService.authService.ensureAuthenticated();
      
      spinner.text = 'Scanning files...';
      const allFiles = await dataSyncService.getAllMapDataFiles();
      
      if (allFiles.length === 0) {
        spinner.warn('No files found to upload');
        return;
      }
      
      // Confirm upload
      spinner.stop();
      const { confirm } = await inquirer.prompt([{
        type: 'confirm',
        name: 'confirm',
        message: `Found ${allFiles.length} files. Do you want to proceed with upload?`,
        default: true
      }]);
      
      if (!confirm) {
        console.log(chalk.yellow('Upload cancelled'));
        return;
      }
      
      spinner.start('Uploading files...');
      
      let result;
      if (options.incremental) {
        result = await dataSyncService.incrementalSync();
      } else {
        result = await dataSyncService.syncAllData();
      }
      
      spinner.succeed(chalk.green('Upload completed!'));
      
      // Display summary
      console.log('\n' + chalk.blue('=== Upload Summary ==='));
      console.log(`Total files: ${result.totalFiles}`);
      console.log(`Uploaded: ${chalk.green(result.uploadedFiles)}`);
      console.log(`Failed: ${chalk.red(result.failedFiles)}`);
      console.log(`Skipped: ${chalk.yellow(result.skippedFiles)}`);
      
    } catch (error) {
      spinner.fail(chalk.red('Upload failed!'));
      console.error(chalk.red('Error:'), error.message);
      process.exit(1);
    }
  });

// List uploaded files command
program
  .command('list')
  .description('List uploaded files on server')
  .option('-t, --type <type>', 'Filter by data type')
  .option('-l, --limit <limit>', 'Limit number of results', '50')
  .action(async (options) => {
    const spinner = ora('Fetching file list...').start();
    
    try {
      const uploadService = new UploadService();
      await uploadService.authService.ensureAuthenticated();
      
      const filters = {};
      if (options.type) filters.type = options.type;
      if (options.limit) filters.limit = parseInt(options.limit);
      
      const files = await uploadService.listUploadedFiles(filters);
      
      spinner.succeed(chalk.green('File list retrieved!'));
      
      if (files.length === 0) {
        console.log(chalk.yellow('No files found'));
        return;
      }
      
      console.log('\n' + chalk.blue('=== Uploaded Files ==='));
      files.forEach((file, index) => {
        console.log(`${index + 1}. ${chalk.cyan(file.fileName)}`);
        console.log(`   Type: ${file.type} | Size: ${formatBytes(file.fileSize)} | Uploaded: ${file.uploadedAt}`);
        console.log('');
      });
      
    } catch (error) {
      spinner.fail(chalk.red('Failed to fetch file list!'));
      console.error(chalk.red('Error:'), error.message);
      process.exit(1);
    }
  });

// Delete file command
program
  .command('delete <fileId>')
  .description('Delete a file from server')
  .option('-f, --force', 'Skip confirmation')
  .action(async (fileId, options) => {
    try {
      if (!options.force) {
        const { confirm } = await inquirer.prompt([{
          type: 'confirm',
          name: 'confirm',
          message: `Are you sure you want to delete file ${fileId}?`,
          default: false
        }]);
        
        if (!confirm) {
          console.log(chalk.yellow('Deletion cancelled'));
          return;
        }
      }
      
      const spinner = ora('Deleting file...').start();
      const uploadService = new UploadService();
      await uploadService.authService.ensureAuthenticated();
      
      await uploadService.deleteFile(fileId);
      
      spinner.succeed(chalk.green('File deleted successfully!'));
      
    } catch (error) {
      spinner.fail(chalk.red('Deletion failed!'));
      console.error(chalk.red('Error:'), error.message);
      process.exit(1);
    }
  });

// Helper function to format bytes
function formatBytes(bytes, decimals = 2) {
  if (bytes === 0) return '0 Bytes';
  
  const k = 1024;
  const dm = decimals < 0 ? 0 : decimals;
  const sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB'];
  
  const i = Math.floor(Math.log(bytes) / Math.log(k));
  
  return parseFloat((bytes / Math.pow(k, i)).toFixed(dm)) + ' ' + sizes[i];
}

// Error handling
process.on('unhandledRejection', (reason, promise) => {
  console.error(chalk.red('Unhandled Rejection at:'), promise, chalk.red('reason:'), reason);
  process.exit(1);
});

// Parse command line arguments
program.parse();
