#!/usr/bin/env node

const chalk = require('chalk');
const ora = require('ora');
const inquirer = require('inquirer');
const path = require('path');
const fs = require('fs-extra');
const DataSyncService = require('../src/services/DataSyncService');

async function uploadCurrentData() {
  console.log(chalk.blue.bold('\n=== Upload Current Map Data ===\n'));
  
  try {
    // Check if we're in the right directory
    const currentDir = process.cwd();
    const parentDir = path.dirname(currentDir);
    
    console.log(chalk.yellow(`Current directory: ${currentDir}`));
    console.log(chalk.yellow(`Parent directory: ${parentDir}`));
    
    // Check if map data exists in parent directory
    const mapDir = path.join(parentDir, 'map');
    const poiDir = path.join(parentDir, 'poi');
    const buildingDir = path.join(parentDir, 'building');
    const speedcamDir = path.join(parentDir, 'speedcam');
    
    const dirsExist = {
      map: await fs.pathExists(mapDir),
      poi: await fs.pathExists(poiDir),
      building: await fs.pathExists(buildingDir),
      speedcam: await fs.pathExists(speedcamDir)
    };
    
    console.log(chalk.blue('\nChecking data directories:'));
    Object.entries(dirsExist).forEach(([dir, exists]) => {
      console.log(`  ${dir}: ${exists ? chalk.green('✓') : chalk.red('✗')}`);
    });
    
    if (!Object.values(dirsExist).some(exists => exists)) {
      console.log(chalk.red('\nNo map data directories found in parent directory!'));
      console.log(chalk.yellow('Please run this script from the zzmaps-api-project directory.'));
      process.exit(1);
    }
    
    // Count files in each directory
    const fileCounts = {};
    for (const [dir, exists] of Object.entries(dirsExist)) {
      if (exists) {
        const dirPath = path.join(parentDir, dir);
        const files = await fs.readdir(dirPath);
        const dataFiles = files.filter(file => 
          ['.fbl', '.fjv', '.fpa', '.fda', '.poi', '.3dc', '.3dl', '.spc', '.spud'].includes(path.extname(file).toLowerCase())
        );
        fileCounts[dir] = dataFiles.length;
      } else {
        fileCounts[dir] = 0;
      }
    }
    
    console.log(chalk.blue('\nFile counts:'));
    Object.entries(fileCounts).forEach(([dir, count]) => {
      console.log(`  ${dir}: ${count} files`);
    });
    
    const totalFiles = Object.values(fileCounts).reduce((sum, count) => sum + count, 0);
    
    if (totalFiles === 0) {
      console.log(chalk.red('\nNo map data files found!'));
      process.exit(1);
    }
    
    console.log(chalk.green(`\nTotal files found: ${totalFiles}`));
    
    // Confirm upload
    const { confirm } = await inquirer.prompt([{
      type: 'confirm',
      name: 'confirm',
      message: 'Do you want to proceed with uploading all map data?',
      default: true
    }]);
    
    if (!confirm) {
      console.log(chalk.yellow('Upload cancelled'));
      return;
    }
    
    // Initialize data sync service
    const dataSyncService = new DataSyncService();
    
    // Update config paths to point to parent directory
    const config = require('../config/config');
    config.paths.mapData = path.join(parentDir, 'map');
    config.paths.poiData = path.join(parentDir, 'poi');
    config.paths.buildingData = path.join(parentDir, 'building');
    config.paths.speedcamData = path.join(parentDir, 'speedcam');
    
    console.log(chalk.blue('\nStarting upload process...'));
    
    // Authenticate
    const spinner = ora('Authenticating...').start();
    try {
      await dataSyncService.uploadService.authService.ensureAuthenticated();
      spinner.succeed(chalk.green('Authentication successful'));
    } catch (error) {
      spinner.fail(chalk.red('Authentication failed'));
      console.error(chalk.red('Error:'), error.message);
      console.log(chalk.yellow('\nPlease check your API credentials in the .env file or environment variables.'));
      process.exit(1);
    }
    
    // Upload data
    spinner.start('Uploading map data...');
    const result = await dataSyncService.syncAllData();
    
    spinner.succeed(chalk.green('Upload completed!'));
    
    // Display summary
    console.log(chalk.blue('\n=== Upload Summary ==='));
    console.log(`Total files processed: ${result.totalFiles}`);
    console.log(`Successfully uploaded: ${chalk.green(result.uploadedFiles)}`);
    console.log(`Failed uploads: ${chalk.red(result.failedFiles)}`);
    console.log(`Skipped files: ${chalk.yellow(result.skippedFiles)}`);
    console.log(`Last sync: ${result.lastSync ? new Date(result.lastSync).toLocaleString() : 'Never'}`);
    
    if (result.failedFiles > 0) {
      console.log(chalk.yellow('\nSome files failed to upload. Check the logs for details.'));
    }
    
    console.log(chalk.green('\nUpload process completed successfully!'));
    
  } catch (error) {
    console.error(chalk.red('\nUpload failed:'), error.message);
    process.exit(1);
  }
}

// Error handling
process.on('unhandledRejection', (reason, promise) => {
  console.error(chalk.red('Unhandled Rejection at:'), promise, chalk.red('reason:'), reason);
  process.exit(1);
});

// Run the script
uploadCurrentData();
