<?php
/**
 * ZZMaps API Client - 1-Click Install
 * Upload this file to your file manager and access via URL
 */

// Set content type
header('Content-Type: text/html; charset=utf-8');

// Check if running on web server
if (php_sapi_name() === 'cli') {
    die("This script must be run via web browser, not command line.\n");
}

// Check if Node.js is available
function checkNodeJS() {
    $output = shell_exec('node --version 2>&1');
    return !empty($output) && strpos($output, 'v') === 0;
}

// Check if npm is available
function checkNPM() {
    $output = shell_exec('npm --version 2>&1');
    return !empty($output) && is_numeric(trim($output));
}

// Install Node.js if not available
function installNodeJS() {
    $commands = [
        'curl -fsSL https://deb.nodesource.com/setup_18.x | sudo -E bash -',
        'sudo apt-get install -y nodejs'
    ];
    
    $output = '';
    foreach ($commands as $cmd) {
        $output .= shell_exec($cmd . ' 2>&1');
    }
    
    return $output;
}

// Install project dependencies
function installDependencies() {
    $output = shell_exec('npm install 2>&1');
    return $output;
}

// Create .env file
function createEnvFile() {
    $envContent = "# ZZMaps API Configuration\n";
    $envContent .= "ZZMAPS_USERNAME=your_username_here\n";
    $envContent .= "ZZMAPS_PASSWORD=your_password_here\n";
    $envContent .= "ZZMAPS_API_KEY=your_api_key_here\n\n";
    $envContent .= "# Optional: Override default settings\n";
    $envContent .= "# API_BASE_URL=https://api.zzmaps.com\n";
    $envContent .= "# API_VERSION=v1\n";
    $envContent .= "# UPLOAD_CHUNK_SIZE=1048576\n";
    $envContent .= "# MAX_CONCURRENT_UPLOADS=3\n";
    
    return file_put_contents('.env', $envContent);
}

// Run project setup
function runSetup() {
    $output = shell_exec('npm run setup 2>&1');
    return $output;
}

// Upload data
function uploadData() {
    $output = shell_exec('npm run upload-current 2>&1');
    return $output;
}

// Get project status
function getProjectStatus() {
    $status = [
        'nodejs' => checkNodeJS(),
        'npm' => checkNPM(),
        'dependencies' => is_dir('node_modules'),
        'env' => file_exists('.env'),
        'package_json' => file_exists('package.json')
    ];
    
    return $status;
}

// Handle form submission
if ($_POST['action'] ?? false) {
    $action = $_POST['action'];
    $result = ['success' => false, 'message' => '', 'output' => ''];
    
    switch ($action) {
        case 'install_nodejs':
            $output = installNodeJS();
            $result['success'] = checkNodeJS();
            $result['message'] = $result['success'] ? 'Node.js installed successfully!' : 'Failed to install Node.js';
            $result['output'] = $output;
            break;
            
        case 'install_dependencies':
            $output = installDependencies();
            $result['success'] = is_dir('node_modules');
            $result['message'] = $result['success'] ? 'Dependencies installed successfully!' : 'Failed to install dependencies';
            $result['output'] = $output;
            break;
            
        case 'create_env':
            $success = createEnvFile();
            $result['success'] = $success !== false;
            $result['message'] = $result['success'] ? '.env file created successfully!' : 'Failed to create .env file';
            break;
            
        case 'run_setup':
            $output = runSetup();
            $result['success'] = true;
            $result['message'] = 'Setup completed!';
            $result['output'] = $output;
            break;
            
        case 'upload_data':
            $output = uploadData();
            $result['success'] = true;
            $result['message'] = 'Data upload completed!';
            $result['output'] = $output;
            break;
    }
    
    header('Content-Type: application/json');
    echo json_encode($result);
    exit;
}

// Get current status
$status = getProjectStatus();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ZZMaps API Client - 1-Click Install</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
            background-color: #f5f5f5;
        }
        .container {
            background: white;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        h1 {
            color: #333;
            text-align: center;
            margin-bottom: 30px;
        }
        .status {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        .status-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 10px 0;
            border-bottom: 1px solid #eee;
        }
        .status-item:last-child {
            border-bottom: none;
        }
        .status-label {
            font-weight: bold;
        }
        .status-value {
            padding: 5px 10px;
            border-radius: 3px;
            font-size: 12px;
            font-weight: bold;
        }
        .status-ok {
            background: #d4edda;
            color: #155724;
        }
        .status-error {
            background: #f8d7da;
            color: #721c24;
        }
        .btn {
            background: #007bff;
            color: white;
            border: none;
            padding: 12px 24px;
            border-radius: 5px;
            cursor: pointer;
            font-size: 16px;
            margin: 5px;
            transition: background 0.3s;
        }
        .btn:hover {
            background: #0056b3;
        }
        .btn:disabled {
            background: #6c757d;
            cursor: not-allowed;
        }
        .btn-success {
            background: #28a745;
        }
        .btn-success:hover {
            background: #1e7e34;
        }
        .btn-warning {
            background: #ffc107;
            color: #212529;
        }
        .btn-warning:hover {
            background: #e0a800;
        }
        .output {
            background: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 5px;
            padding: 15px;
            margin-top: 20px;
            font-family: monospace;
            font-size: 12px;
            white-space: pre-wrap;
            max-height: 300px;
            overflow-y: auto;
            display: none;
        }
        .step {
            margin: 20px 0;
            padding: 20px;
            border: 1px solid #dee2e6;
            border-radius: 5px;
        }
        .step h3 {
            margin-top: 0;
            color: #495057;
        }
        .alert {
            padding: 15px;
            margin: 20px 0;
            border-radius: 5px;
        }
        .alert-info {
            background: #d1ecf1;
            color: #0c5460;
            border: 1px solid #bee5eb;
        }
        .alert-warning {
            background: #fff3cd;
            color: #856404;
            border: 1px solid #ffeaa7;
        }
        .alert-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🚀 ZZMaps API Client - 1-Click Install</h1>
        
        <div class="alert alert-info">
            <strong>Hướng dẫn:</strong> Upload dự án lên file manager, sau đó truy cập URL này để cài đặt tự động. 
            Không cần quyền SSH hay VPS!
        </div>
        
        <div class="status">
            <h3>📊 Trạng thái dự án</h3>
            <div class="status-item">
                <span class="status-label">Node.js</span>
                <span class="status-value <?php echo $status['nodejs'] ? 'status-ok' : 'status-error'; ?>">
                    <?php echo $status['nodejs'] ? '✓ Installed' : '✗ Not Installed'; ?>
                </span>
            </div>
            <div class="status-item">
                <span class="status-label">npm</span>
                <span class="status-value <?php echo $status['npm'] ? 'status-ok' : 'status-error'; ?>">
                    <?php echo $status['npm'] ? '✓ Installed' : '✗ Not Installed'; ?>
                </span>
            </div>
            <div class="status-item">
                <span class="status-label">Dependencies</span>
                <span class="status-value <?php echo $status['dependencies'] ? 'status-ok' : 'status-error'; ?>">
                    <?php echo $status['dependencies'] ? '✓ Installed' : '✗ Not Installed'; ?>
                </span>
            </div>
            <div class="status-item">
                <span class="status-label">Configuration</span>
                <span class="status-value <?php echo $status['env'] ? 'status-ok' : 'status-error'; ?>">
                    <?php echo $status['env'] ? '✓ Created' : '✗ Not Created'; ?>
                </span>
            </div>
            <div class="status-item">
                <span class="status-label">Project Files</span>
                <span class="status-value <?php echo $status['package_json'] ? 'status-ok' : 'status-error'; ?>">
                    <?php echo $status['package_json'] ? '✓ Ready' : '✗ Missing'; ?>
                </span>
            </div>
        </div>
        
        <div class="step">
            <h3>🔧 Bước 1: Cài đặt Node.js</h3>
            <p>Node.js là môi trường chạy JavaScript cần thiết cho dự án.</p>
            <button class="btn" onclick="runAction('install_nodejs')" <?php echo $status['nodejs'] ? 'disabled' : ''; ?>>
                <?php echo $status['nodejs'] ? '✓ Node.js đã cài đặt' : 'Cài đặt Node.js'; ?>
            </button>
        </div>
        
        <div class="step">
            <h3>📦 Bước 2: Cài đặt Dependencies</h3>
            <p>Cài đặt các thư viện JavaScript cần thiết cho dự án.</p>
            <button class="btn" onclick="runAction('install_dependencies')" <?php echo !$status['nodejs'] || $status['dependencies'] ? 'disabled' : ''; ?>>
                <?php echo $status['dependencies'] ? '✓ Dependencies đã cài đặt' : 'Cài đặt Dependencies'; ?>
            </button>
        </div>
        
        <div class="step">
            <h3>⚙️ Bước 3: Tạo file cấu hình</h3>
            <p>Tạo file .env để lưu thông tin API credentials.</p>
            <button class="btn" onclick="runAction('create_env')" <?php echo $status['env'] ? 'disabled' : ''; ?>>
                <?php echo $status['env'] ? '✓ File cấu hình đã tạo' : 'Tạo file cấu hình'; ?>
            </button>
        </div>
        
        <div class="step">
            <h3>🚀 Bước 4: Chạy Setup</h3>
            <p>Thiết lập dự án và nhập thông tin API credentials.</p>
            <button class="btn" onclick="runAction('run_setup')" <?php echo !$status['dependencies'] || !$status['env'] ? 'disabled' : ''; ?>>
                Chạy Setup
            </button>
        </div>
        
        <div class="step">
            <h3>📤 Bước 5: Upload dữ liệu</h3>
            <p>Upload 461MB dữ liệu bản đồ Việt Nam lên api.zzmaps.com.</p>
            <button class="btn btn-success" onclick="runAction('upload_data')" <?php echo !$status['dependencies'] ? 'disabled' : ''; ?>>
                Upload dữ liệu
            </button>
        </div>
        
        <div class="output" id="output"></div>
        
        <div class="alert alert-warning">
            <strong>Lưu ý:</strong> Sau khi upload dữ liệu, bạn cần chỉnh sửa file .env để nhập thông tin API credentials thực tế.
        </div>
        
        <div class="alert alert-success">
            <strong>Hoàn thành!</strong> Sau khi cài đặt xong, dữ liệu bản đồ sẽ được lưu trên api.zzmaps.com và sẵn sàng cho ứng dụng mobile sử dụng.
        </div>
    </div>
    
    <script>
        function runAction(action) {
            const output = document.getElementById('output');
            const button = event.target;
            
            // Disable button
            button.disabled = true;
            button.textContent = 'Đang chạy...';
            
            // Show output
            output.style.display = 'block';
            output.textContent = 'Đang chạy...';
            
            // Send request
            fetch('', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'action=' + action
            })
            .then(response => response.json())
            .then(data => {
                output.textContent = data.output || data.message;
                
                if (data.success) {
                    button.textContent = '✓ Hoàn thành';
                    button.classList.add('btn-success');
                } else {
                    button.textContent = '✗ Lỗi';
                    button.classList.add('btn-warning');
                }
            })
            .catch(error => {
                output.textContent = 'Lỗi: ' + error.message;
                button.textContent = '✗ Lỗi';
                button.classList.add('btn-warning');
            });
        }
    </script>
</body>
</html>
